"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var RpiWebClient = (function () {
    function RpiWebClient() {
        var _this = this;
        this.defaults = {};
        this.realtimeParameters = [];
        this.config = {};
        this.visitor = null;
        this.LOAD_EVENT = "rpiWebClientLoad";
        this.PRE_INIT_EVENT = "rpiWebClientPreInit";
        this.INIT_EVENT = "rpiWebClientInit";
        this.RENDER_CONTEXT_EVENT = "rpiWebClientContextDecisionRender";
        this.RENDER_CONTENT_EVENT = "rpiWebClientDecisionRender";
        this.RENDER_LOOKUP_EVENT = "rpiWebClientLookupRequestRender";
        this.VISITOR_SET_EVENT = "rpiWebClientVisitorSet";
        this.VISITOR_CHANGE_EVENT = "rpiWebClientVisitorChange";
        this.DECISION_RESULT_CLICK_EVENT = "rpiDecisionResultLinkClick";
        if (rpiWebClientDefaults) {
            this.defaults = rpiWebClientDefaults;
        }
        this.decisionOptions = {};
        this.pageParameters = {
            pageTitle: "",
            goals: [],
            pendingEvents: [],
            impressionId: "",
            cId: "",
            exId: 0,
            geolocation: null,
            pagePublishId: "0"
        };
        this.urlParams = this.serializeQueryString();
        this.parseRpiPageParams();
        var self = this;
        setTimeout(function () {
            self.publishEvent(_this.LOAD_EVENT, self);
        }, 0);
    }
    RpiWebClient.prototype.init = function (opts, callback) {
        var self = this;
        if (!opts) {
            opts = {};
        }
        this.config = __assign({}, this.defaults, opts);
        this.log("config: " + JSON.stringify(this.config), "INFO");
        this.publishEvent(this.PRE_INIT_EVENT, this);
        this.visitor = this.getVisitor();
        this.log("visitor: " + JSON.stringify(this.visitor), "INFO");
        this.registerForClickEvents();
        if (this.config.requestGeolocation) {
            this.requestGeolocation(function (position) {
                if (position.coords) {
                    this.pageParameters.geolocation = position;
                    this.flushRealtimeParameters();
                }
            });
        }
        this.subscribeEvent(window, this.VISITOR_CHANGE_EVENT, function (event) {
            self.log("Handling " + self.VISITOR_CHANGE_EVENT + " event... pushing PageVisit web event.");
            var pageTitle = document.title;
            self.pushWebEvent('PageVisit', null, pageTitle);
        });
        this.submitVisitorDetails(function (data) {
            if (this.config.contexts) {
                this.renderContextDecisions(this.config.contexts);
            }
            if (this.config.contentDecisions) {
                this.renderDecisions(this.config.contentDecisions);
            }
            this.publishEvent(this.INIT_EVENT, this);
            if (callback) {
                callback();
            }
        });
    };
    RpiWebClient.prototype.log = function (message, logLevel) {
        if (!logLevel) {
            logLevel = "LOG";
        }
        if (this.config.debug) {
            switch (logLevel) {
                case "INFO":
                    console.info(message);
                    return;
                case "WARN":
                    console.warn(message);
                    return;
                case "ERROR":
                    console.error(message);
                    return;
            }
            console.log(message);
        }
    };
    ;
    RpiWebClient.prototype.subscribeEvent = function (element, type, callback) {
        try {
            element.addEventListener(type, callback, false);
        }
        catch (e) {
            element.attachEvent('on' + type, callback);
        }
    };
    ;
    RpiWebClient.prototype.unsubscribeElement = function (element, type, callback) {
        if (element.removeEventListener) {
            element.removeEventListener(type, callback, false);
        }
        else if (element.detachEvent) {
            element.detachEvent("on" + type, callback);
        }
        else {
            element["on" + type] = null;
        }
    };
    RpiWebClient.prototype.publishEvent = function (type, detail) {
        if (document.createEvent) {
            var event_1;
            try {
                event_1 = new CustomEvent(type, { detail: detail });
            }
            catch (e) {
                event_1 = document.createEvent("CustomEvent");
                event_1.initCustomEvent(type, false, false, detail);
            }
            window.dispatchEvent(event_1);
        }
        else {
            var event_2 = document.createEventObject();
            event_2.detail = detail;
            window.fireEvent('on' + type, event_2);
        }
    };
    ;
    RpiWebClient.prototype.triggerClickMouseEvent = function (element) {
        var event;
        try {
            event = new MouseEvent('click', {
                bubbles: true,
                cancelable: true,
                view: window
            });
        }
        catch (e) {
            event = document.createEvent("MouseEvent");
            event.initMouseEvent("click", true, true, window, 0, 0, 0, 0, 0, false, false, false, false, 0, null);
        }
        element.dispatchEvent(event);
    };
    ;
    RpiWebClient.prototype.serializeQueryString = function (queryString) {
        if (!queryString) {
            queryString = location.search.slice(1);
        }
        if (!queryString.length) {
            return null;
        }
        var pairs = queryString.split('&');
        var result = {};
        for (var _i = 0, pairs_1 = pairs; _i < pairs_1.length; _i++) {
            var pair = pairs_1[_i];
            var parts = pair.split('=');
            var key = parts[0].toLowerCase();
            if (result[key]) {
                result[key] = decodeURIComponent(result[key]) + "," + decodeURIComponent(parts[1] || '');
            }
            else {
                result[decodeURIComponent(key)] = decodeURIComponent(parts[1] || '');
            }
        }
        return result;
    };
    ;
    RpiWebClient.prototype.setElementData = function (element, name, value) {
        if (element.dataset) {
            element.dataset[name] = value;
        }
        else {
            element.setAttribute("data-" + name, value);
        }
    };
    RpiWebClient.prototype.getElementData = function (element, name) {
        if (element.dataset) {
            var value = element.dataset[name];
            if (value) {
                return value;
            }
        }
        else {
            var value = element.getAttribute("data-" + name);
            if (value) {
                return value;
            }
        }
        return "";
    };
    RpiWebClient.prototype.detectBrowser = function () {
        var browser = "OTHER";
        if (navigator.userAgent.indexOf("Edge") > -1) {
            return "EDGE";
        }
        else if (navigator.userAgent.indexOf("OPR") > -1) {
            return "OPERA";
        }
        else if (navigator.userAgent.indexOf("Chrome") > -1) {
            return "CHROME";
        }
        else if (navigator.userAgent.indexOf("Safari") > -1) {
            return "SAFARI";
        }
        else if (navigator.userAgent.indexOf("Firefox") > -1) {
            return "FIREFOX";
        }
        else if (navigator.userAgent.indexOf("MSIE") > -1 || navigator.userAgent.indexOf("Trident") > -1) {
            return "MSIE";
        }
        return browser;
    };
    RpiWebClient.prototype.parseRpiPageParams = function () {
        if (this.urlParams) {
            if (this.urlParams["exid"]) {
                this.pageParameters.exId = Number(this.urlParams["exid"]);
            }
            if (this.urlParams["rpcid"]) {
                this.pageParameters.cId = this.urlParams["rpcid"];
            }
        }
    };
    RpiWebClient.prototype.submitVisitorDetails = function (callback) {
        this.log("Submitting visitor details...", "INFO");
        var pageTitle = document.title;
        if (this.pageParameters.pageTitle) {
            pageTitle = this.pageParameters.pageTitle;
        }
        this.submitVisitorRequestAsync(function (data) {
            if (callback) {
                callback.call(this, data);
            }
            this.pushWebEvent('PageVisit', null, pageTitle);
        });
    };
    RpiWebClient.prototype.getContextDecision = function (context, callback, slotName, returnAll) {
        this.log("Getting decisions for context: " + context, "INFO");
        var slotNameParam = "";
        if (slotName)
            slotNameParam = "&slotName=" + slotName;
        if (returnAll == undefined) {
            returnAll = true;
        }
        var self = this;
        if (this.visitor) {
            var deviceIDParam = "";
            if (this.visitor.deviceId) {
                deviceIDParam = "&deviceID=" + this.visitor.deviceId;
            }
            var url = this.getDecisionsContextsUrl() + this.config.clientId +
                "/" + context + "/" + this.visitor.visitorId +
                "?returnAll=" + returnAll +
                "&minuteOffSetUTC=" + encodeURIComponent(new Date().getTimezoneOffset().toString()) +
                deviceIDParam + slotNameParam;
            this.requestAsync(url, function (data) {
                callback.call(self, this.parseDecisionResponse(data));
            });
        }
        else {
            this.log("Unable to get context decision, visitor not set", "WARN");
        }
    };
    RpiWebClient.prototype.renderContextDecision = function (context) {
        this.log("Rendering content for context: " + context, "INFO");
        this.getContextDecision(context, function (results) {
            for (var _i = 0, results_1 = results; _i < results_1.length; _i++) {
                var result = results_1[_i];
                this.renderDecisionContent(result);
            }
            this.publishEvent(this.RENDER_CONTEXT_EVENT, results);
        });
    };
    RpiWebClient.prototype.renderContextDecisions = function (contexts) {
        this.log("Rendering content for contexts: " + JSON.stringify(contexts), "INFO");
        for (var i = 0; i < contexts.length; i++) {
            this.renderContextDecision(contexts[i]);
        }
    };
    RpiWebClient.prototype.getDecision = function (contentId, callback) {
        this.log("Getting decision for content id: " + contentId, "INFO");
        var self = this;
        if (this.visitor) {
            var deviceIDParam = "";
            if (this.visitor.deviceId)
                deviceIDParam = "&deviceID=" + this.visitor.deviceId;
            var url = this.getDecisionsUrl() + contentId + "/" + this.visitor.visitorId +
                "?publishID=" + this.pageParameters.pagePublishId + deviceIDParam + "&isMasterMessage=false" +
                "&minuteOffSetUTC=" + encodeURIComponent(new Date().getTimezoneOffset().toString());
            this.requestAsync(url, function (data) {
                callback.call(self, this.parseDecisionResponse(data));
            });
        }
        else {
            this.log("Unable to get decision, visitor not set", "WARN");
        }
    };
    RpiWebClient.prototype.renderDecision = function (contentId, elementId) {
        this.log("Rendering decision for content id: " + contentId, "INFO");
        this.getDecision(contentId, function (result) {
            this.renderDecisionContent(result, elementId);
            this.publishEvent(this.RENDER_CONTENT_EVENT, result);
        });
    };
    RpiWebClient.prototype.renderDecisions = function (contentDecisions) {
        this.log("Rendering content decisions: " + JSON.stringify(contentDecisions), "INFO");
        for (var _i = 0, contentDecisions_1 = contentDecisions; _i < contentDecisions_1.length; _i++) {
            var contentDecision = contentDecisions_1[_i];
            this.renderDecision(contentDecision.contentId, contentDecision.elementId);
        }
    };
    RpiWebClient.prototype.getVisitorView = function (viewName, callback) {
        this.log("Getting visitor view: " + viewName, "INFO");
        var self = this;
        if (this.visitor) {
            var url = this.getVisitorViewUrl() + viewName;
            var visitorViewArguments = {
                Identity: { VisitorID: this.visitor.visitorId },
                Attributes: this.getRealtimeParameters()
            };
            this.postRequestAsync(url, JSON.stringify(visitorViewArguments), function (data) {
                callback.call(this, this.parseDecisionResponse(data));
            });
        }
        else {
            this.log("Unable to get visitor view, visitor not set", "WARN");
        }
    };
    RpiWebClient.prototype.getLookupAttributes = function (lookupRequest, callback) {
        this.log("Getting lookup attributes for request: " + lookupRequest, "INFO");
        if (this.visitor) {
            var url = this.getCacheVisitorsAttributesUrl() + this.visitor.visitorId;
            this.postRequestAsync(url, lookupRequest, function (data) {
                callback.call(this, this.parseDecisionResponse(data));
            });
        }
        else {
            this.log("Unable to get lookup attributes, visitor not set", "WARN");
        }
    };
    RpiWebClient.prototype.renderLookupAttributes = function (lookupRequest) {
        this.log("Rendering lookup attributes for request: " + lookupRequest, "INFO");
        this.getLookupAttributes(lookupRequest, function (results) {
            for (var _i = 0, results_2 = results; _i < results_2.length; _i++) {
                var result = results_2[_i];
                var elements = document.getElementsByName(result.ID);
                for (var i = 0; i < elements.length; i++) {
                    elements[i].innerHTML = result.Value;
                }
            }
            this.publishEvent(this.RENDER_LOOKUP_EVENT, results);
        });
    };
    RpiWebClient.prototype.submitVisitorRequestAsync = function (onSuccess, onError) {
        var url = this.getCacheVisitUrl();
        var regDetails = this.getVisitorRegistrationDetails();
        this.postRequestAsync(url, JSON.stringify(regDetails), function (data) {
            this.parseVisitor(data, true);
            if (onSuccess) {
                onSuccess.call(this, data);
            }
        }, onError);
    };
    RpiWebClient.prototype.parseDecisionResponse = function (data) {
        this.log("Parsing response data: " + data, "INFO");
        var decisions = JSON.parse(data);
        return decisions;
    };
    RpiWebClient.prototype.registerForClickEvents = function (enableLinkTracking) {
        var self = this;
        if (enableLinkTracking == undefined) {
            if (this.config.enableClickTracking == undefined) {
                enableLinkTracking = false;
            }
            else {
                enableLinkTracking = this.config.enableClickTracking;
            }
        }
        if (!enableLinkTracking) {
            return;
        }
        var elements = document.getElementsByTagName('a');
        for (var i = 0, len = elements.length; i < len; i++) {
            this.subscribeEvent(elements[i], "click", this.webEventClickHandler);
        }
    };
    RpiWebClient.prototype.webEventClickHandler = function (event) {
        rpiWebClient.unsubscribeElement(event.target, event.type, rpiWebClient.webEventClickHandler);
        if (event.target) {
            var element_1 = event.target;
            if (element_1.nodeName == "A") {
                var anchor = element_1;
                var metadata = [];
                var resultId = rpiWebClient.getElementData(element_1, "resultId");
                if (resultId) {
                    rpiWebClient.publishEvent(rpiWebClient.DECISION_RESULT_CLICK_EVENT, resultId);
                    metadata = [{ Name: "resultId", Value: resultId }];
                }
                rpiWebClient.sendMetric("PageLinkClick", null, anchor.href, null, metadata, function (data) {
                    rpiWebClient.triggerClickMouseEvent(element_1);
                });
                var timeoutThreshold = rpiWebClient.getClickEventTimeoutThreshold();
                setTimeout(function () {
                    rpiWebClient.triggerClickMouseEvent(element_1);
                }, timeoutThreshold);
                event.preventDefault();
            }
        }
    };
    RpiWebClient.prototype.getClickEventTimeoutThreshold = function () {
        var timeoutThreshold = 0;
        var browser = this.detectBrowser();
        if (window.rpiRealtimeClickWebEventTimeout != undefined) {
            timeoutThreshold = window.rpiRealtimeClickWebEventTimeout;
        }
        else if (browser == "FIREFOX" || browser == "SAFARI" || browser == "OTHER") {
            timeoutThreshold = 100;
        }
        return timeoutThreshold;
    };
    RpiWebClient.prototype.getRealtimeParameters = function () {
        var exclList = ["rpcid", "exid", "rpimg"];
        var params = [];
        if (this.urlParams) {
            if (this.config.enableUrlParameterCaching) {
                for (var key in this.urlParams) {
                    var value = this.urlParams[key];
                    if (exclList.indexOf(key) < 0) {
                        params.push({ Name: key, Value: value });
                    }
                }
            }
            else {
                if (this.config.urlParameterCachingOverrides && this.config.urlParameterCachingOverrides.length > 0) {
                    for (var _i = 0, _a = this.config.urlParameterCachingOverrides; _i < _a.length; _i++) {
                        var param = _a[_i];
                        var key = param.toLocaleLowerCase();
                        if (this.urlParams[key]) {
                            params.push({ Name: key, Value: this.urlParams[key] });
                        }
                    }
                }
            }
        }
        if (this.realtimeParameters.length > 0) {
            params = params.concat(this.realtimeParameters);
            this.realtimeParameters = [];
        }
        return params;
    };
    RpiWebClient.prototype.getVisitorRegistrationDetails = function () {
        var publishId = "0";
        if (this.pageParameters.pagePublishId) {
            publishId = this.pageParameters.pagePublishId;
        }
        var exID = null;
        var cID = null;
        var msgIDs = null;
        if (this.urlParams) {
            exID = this.urlParams["exid"];
            cID = this.urlParams["rpcid"];
            msgIDs = this.urlParams["rpimg"];
        }
        var visitDetails = {};
        visitDetails["PagePublishedID"] = publishId;
        if (!this.visitor) {
            visitDetails["IsNewVisitor"] = true;
        }
        else {
            visitDetails["IsNewVisitor"] = false;
            visitDetails["VisitorID"] = this.visitor.visitorId;
            visitDetails["DeviceID"] = this.visitor.deviceId;
        }
        var trackingObjects = {};
        if (exID) {
            trackingObjects["ChannelExecutionID"] = exID;
            if (cID) {
                trackingObjects["RPContactID"] = cID;
            }
        }
        if (msgIDs) {
            var msgComp = msgIDs.split("_");
            if (msgComp.length > 1) {
                trackingObjects["MessageListID"] = msgComp[0];
                trackingObjects["MessageID"] = msgComp[1];
            }
        }
        visitDetails["InteractionTracking"] = trackingObjects;
        if (this.config.clientId) {
            visitDetails["ClientID"] = this.config.clientId;
        }
        var referrer = document.referrer;
        if (referrer) {
            visitDetails["PageReferrer"] = referrer;
        }
        if (this.pageParameters.geolocation) {
            visitDetails["Geolocation"] = {
                "Longitude": this.pageParameters.geolocation.coords.longitude,
                "Latitude": this.pageParameters.geolocation.coords.latitude
            };
        }
        visitDetails["VisitorAttributes"] = this.getRealtimeParameters();
        return visitDetails;
    };
    RpiWebClient.prototype.requestGeolocation = function (onSuccess) {
        if (!this.getGeolocationCookie()) {
            this.setGeolocationCookie();
            if (navigator.geolocation) {
                var self_1 = this;
                navigator.geolocation.getCurrentPosition(function (position) {
                    self_1.log("Geolocation request success: longitude: " + position.coords.longitude + " latitude: " + position.coords.latitude, "INFO");
                    onSuccess.call(self_1, position);
                }, function (error) {
                    self_1.log("Geolocation request failed with error code: " + error.code + ", message: " + error.message, "WARN");
                }, { maximumAge: 300000, timeout: 10000 });
            }
            else {
                this.log("Geolocation not supported by this browser", "WARN");
            }
        }
    };
    RpiWebClient.prototype.pushWebEvent = function (state, stateValue, extraParam, contentId, metadata) {
        if (!extraParam) {
            extraParam = null;
        }
        if (!contentId) {
            contentId = null;
        }
        if (!this.visitor) {
            var rpiEvent = { state: state, stateValue: stateValue, extraParam: extraParam, contentID: contentId, metadata: metadata };
            this.pageParameters.pendingEvents.push(rpiEvent);
        }
        else {
            this.sendMetric(state, stateValue, extraParam, contentId, metadata);
            this.flushPendingEvents();
        }
    };
    RpiWebClient.prototype.flushPendingEvents = function () {
        if (this.visitor) {
            var hiddenElements = document.getElementsByName('rpiVisitorID');
            for (var i = 0; i < hiddenElements.length; i++) {
                hiddenElements[i].value = this.visitor.visitorId;
            }
            hiddenElements = document.getElementsByName('rpiDeviceID');
            for (var i = 0; i < hiddenElements.length; i++) {
                hiddenElements[i].value = this.visitor.deviceId;
            }
        }
        if (this.pageParameters.impressionId) {
            var hiddenElements = document.getElementsByName('rpiImpID');
            for (var i = 0; i < hiddenElements.length; i++) {
                hiddenElements[i].value = this.pageParameters.impressionId;
            }
        }
        if (this.pageParameters.pendingEvents) {
            for (var i = 0; i < this.pageParameters.pendingEvents.length; i++) {
                var event_3 = this.pageParameters.pendingEvents[i];
                this.sendMetric(event_3["state"], event_3["stateValue"], event_3["extraParam"], event_3["contentID"], event_3["metadata"]);
            }
            this.pageParameters.pendingEvents = [];
        }
    };
    RpiWebClient.prototype.sendMetric = function (state, stateValue, extraParam, contentID, metadata, callback) {
        var url = this.getEventsUrl();
        var eventDetails = {};
        eventDetails["EventName"] = state;
        if (this.pageParameters.exId > 0) {
            eventDetails["ChannelExecutionID"] = this.pageParameters.exId;
        }
        if (this.pageParameters.cId) {
            eventDetails["RPContactID"] = this.pageParameters.cId;
        }
        if (stateValue) {
            eventDetails["MetricValue"] = stateValue;
        }
        if (extraParam) {
            eventDetails["EventDetail"] = extraParam;
        }
        var publishId = "0";
        if (this.pageParameters.pagePublishId) {
            publishId = this.pageParameters.pagePublishId;
        }
        eventDetails["PagePublishedID"] = publishId;
        if (this.visitor) {
            eventDetails["VisitorID"] = this.visitor.visitorId;
            eventDetails["DeviceID"] = this.visitor.deviceId;
        }
        if (this.pageParameters.impressionId)
            eventDetails["ImpressionID"] = this.pageParameters.impressionId;
        if (this.config.clientId) {
            eventDetails["ClientID"] = this.config.clientId;
        }
        if (contentID) {
            eventDetails["ContentID"] = contentID;
        }
        if (state == 'PageVisit') {
            if (this.pageParameters.exId > 0) {
                eventDetails["PageReferral"] = "exid" + this.pageParameters.exId;
            }
            else {
                var referrer = document.referrer;
                if (referrer) {
                    eventDetails["PageReferral"] = referrer;
                }
                else {
                    eventDetails["PageReferral"] = "Direct";
                }
            }
        }
        if (this.pageParameters.goals && this.pageParameters.goals.length > 0) {
            var goalDetail = {};
            for (var _i = 0, _a = this.pageParameters.goals; _i < _a.length; _i++) {
                var goal = _a[_i];
                if (goal["state"] && goal["state"].toLowerCase() == state.toLowerCase()) {
                    if (goal["detail"]) {
                        goalDetail["GoalContentID"] = goal["content"];
                        if (goal["name"]) {
                            goalDetail["GoalName"] = goal["name"];
                        }
                        eventDetails["GoalDetails"] = goalDetail;
                    }
                    else if (extraParam && extraParam.toLowerCase().indexOf(goal["detail"].toLowerCase()) >= 0) {
                        goalDetail["GoalContentID"] = goal["content"];
                        if (goal["name"]) {
                            goalDetail["GoalName"] = goal["name"];
                        }
                        eventDetails["GoalDetails"] = goalDetail;
                    }
                }
            }
        }
        if (metadata && metadata.length > 0) {
            eventDetails["Metadata"] = metadata;
        }
        this.postRequestAsync(url, JSON.stringify(eventDetails), function (data) {
            if (callback) {
                callback.call(this, data);
            }
        });
    };
    RpiWebClient.prototype.renderDecisionContent = function (result, elementId, lookupRequest) {
        if (!elementId) {
            elementId = this.getElementId(result);
        }
        var element = document.getElementById(elementId);
        if (!element) {
            this.log("Unable to find DOM element with ID: " + elementId, "WARN");
            return;
        }
        this.getDecisionContent(result, function (content) {
            if (element) {
                if (result.ContentFormat == "HTML") {
                    element.innerHTML = "";
                    var elements = this.parseHtml(content);
                    for (var i = 0; i < elements.length; i++) {
                        element.appendChild(elements[i]);
                    }
                }
                else {
                    element.innerHTML = content;
                }
                if (result.ResultID) {
                    var anchors = element.getElementsByTagName("a");
                    for (var i = 0; i < anchors.length; i++) {
                        var anchor = anchors[i];
                        this.setElementData(anchor, "resultId", result.ResultID);
                        this.subscribeEvent(anchor, "click", this.webEventClickHandler);
                    }
                }
            }
        }, lookupRequest);
    };
    RpiWebClient.prototype.getElementId = function (result) {
        if (result.SlotName) {
            return result.SlotName;
        }
        return result.DivName;
    };
    RpiWebClient.prototype.parseHtml = function (content) {
        var elements = [];
        var body = document.createElement("body");
        body.innerHTML = content;
        for (var i = 0; i < body.children.length; i++) {
            var child = body.children[i];
            if (child.nodeName == "SCRIPT") {
                var script = document.createElement("script");
                for (var i_1 in child.attributes) {
                    var attrib = child.attributes[i_1];
                    script.setAttribute(attrib.name, attrib.value);
                }
                script.innerHTML = child.innerHTML;
                elements.push(script);
            }
            else {
                elements.push(child);
            }
        }
        return elements;
    };
    RpiWebClient.prototype.getDecisionContent = function (result, callback, lookupRequest) {
        if (result.IsEmptyResult || result.Result == "0") {
            return;
        }
        var self = this;
        if (result.IsCachedContent) {
            callback.call(self, result.ResultContent);
            return;
        }
        this.requestAsync(result.ContentPath, function (data) {
            callback.call(this, data);
            if (lookupRequest && data.indexOf("-RPIEMBA-") >= 0) {
                this.renderLookupAttributes(lookupRequest);
            }
        });
    };
    RpiWebClient.prototype.pushRealtimeParameter = function (name, value) {
        this.log("Pushing name: " + name + ", value: " + value + " to realtimeParameters...", "INFO");
        this.realtimeParameters.push({ Name: name, Value: value });
    };
    RpiWebClient.prototype.flushRealtimeParameters = function (callback) {
        this.log("Pushing realtimeParameters to visitor profile: " + JSON.stringify(this.realtimeParameters), "INFO");
        var url = this.getCacheVisitUrl();
        var regDetails = {};
        if (!this.visitor) {
            regDetails["IsNewVisitor"] = true;
        }
        else {
            regDetails["IsNewVisitor"] = false;
            regDetails["VisitorID"] = this.visitor.visitorId;
            regDetails["DeviceID"] = this.visitor.deviceId;
        }
        if (this.realtimeParameters && this.realtimeParameters.length) {
            regDetails["VisitorAttributes"] = this.realtimeParameters;
        }
        if (this.pageParameters.geolocation) {
            regDetails["Geolocation"] = {
                "Longitude": this.pageParameters.geolocation.coords.longitude,
                "Latitude": this.pageParameters.geolocation.coords.latitude
            };
        }
        if (regDetails["VisitorAttributes"] || regDetails["Geolocation"]) {
            this.postRequestAsync(url, JSON.stringify(regDetails), function (data) {
                this.parseVisitor(data, true);
                if (callback) {
                    callback.call(this, data);
                }
            });
        }
    };
    RpiWebClient.prototype.request = function (url) {
        this.log("Performing GET request to: " + url);
        var xmlHttp = this.getXhr();
        xmlHttp.open("GET", url, false);
        xmlHttp.setRequestHeader("Accept", "application/json");
        xmlHttp.send();
        return xmlHttp;
    };
    RpiWebClient.prototype.postRequest = function (url, contents) {
        this.log("Performing POST request to: " + url + " with contents: " + contents);
        var xmlHttp = this.getXhr();
        xmlHttp.open("POST", url, false);
        xmlHttp.setRequestHeader("Accept", "application/json");
        xmlHttp.setRequestHeader("Content-type", "text/plain");
        xmlHttp.send(contents);
        return xmlHttp;
    };
    RpiWebClient.prototype.requestAsync = function (url, onSuccess, onError) {
        this.log("Performing async GET request to " + url, "INFO");
        var xmlHttp = this.getXhr();
        var self = this;
        xmlHttp.onreadystatechange = function () {
            if (this.readyState === 4) {
                if (this.status === 200 && typeof onSuccess == 'function') {
                    onSuccess.call(self, this.responseText);
                }
                else if (typeof onError == 'function') {
                    this.log("Error performing async GET:" + this.statusText, "ERROR");
                    onError.call(self, this.statusText);
                }
            }
        };
        xmlHttp.open("GET", url, true);
        xmlHttp.send();
    };
    RpiWebClient.prototype.postRequestAsync = function (url, contents, onSuccess, onError) {
        this.log("Performing async POST request to: " + url + " with contents: " + contents, "INFO");
        var xmlHttp = this.getXhr();
        var self = this;
        xmlHttp.onreadystatechange = function () {
            if (this.readyState === 4) {
                if (this.status === 200 || this.status === 204) {
                    if (onSuccess) {
                        onSuccess.call(self, this.responseText);
                    }
                }
                else if (onError) {
                    this.log("Error performing async POST:" + this.statusText, "ERROR");
                    onError.call(self, this.responseText);
                }
            }
        };
        xmlHttp.open("POST", url, true);
        xmlHttp.setRequestHeader("Accept", "application/json");
        xmlHttp.setRequestHeader("Content-type", "text/plain");
        xmlHttp.send(contents);
    };
    RpiWebClient.prototype.getCookie = function (cookieName) {
        var cookies = document.cookie.split(";");
        for (var _i = 0, cookies_1 = cookies; _i < cookies_1.length; _i++) {
            var cookie = cookies_1[_i];
            var name_1 = cookie.substr(0, cookie.indexOf("="));
            var value = cookie.substr(cookie.indexOf("=") + 1);
            name_1 = name_1.replace(/^\s+|\s+$/g, "");
            if (name_1 == cookieName) {
                return unescape(value);
            }
        }
        return null;
    };
    RpiWebClient.prototype.setCookie = function (cookieName, cookieValue, days, domain, expiry) {
        if (!expiry) {
            expiry = new Date();
            if (days) {
                expiry.setDate(expiry.getDate() + days);
            }
        }
        var value = cookieName + "=" + escape(cookieValue)
            + ((!expiry) ? "" : "; expires=" + expiry.toUTCString())
            + ((!domain) ? "" : "; domain=" + domain)
            + "; path=/";
        this.log("setting cookie: " + value, "INFO");
        document.cookie = value;
    };
    RpiWebClient.prototype.getVisitor = function () {
        var details = this.getCookie(this.config.clientId);
        if (!details)
            return null;
        var visitorId = details;
        var deviceId = "";
        details = this.getCookie(this.config.clientId + "_d");
        if (details) {
            deviceId = details;
        }
        var visitor = { visitorId: visitorId, deviceId: deviceId };
        return visitor;
    };
    RpiWebClient.prototype.setVisitor = function (apiVisitor) {
        if (this.config.clientId) {
            var visitorSet = false;
            var visitorChanged = false;
            if (!this.visitor) {
                visitorSet = true;
            }
            else if (this.visitor.visitorId != apiVisitor.VisitorID) {
                visitorChanged = true;
            }
            this.setCookie(this.config.clientId, apiVisitor.VisitorID, this.config.visitorCookieDuration, this.config.realtimeCookieDomain);
            if (apiVisitor.DeviceID) {
                this.setCookie(this.config.clientId + "_d", apiVisitor.DeviceID, this.config.visitorCookieDuration, this.config.realtimeCookieDomain);
            }
            this.visitor = { visitorId: apiVisitor.VisitorID, deviceId: apiVisitor.DeviceID };
            if (visitorSet) {
                this.publishEvent(this.VISITOR_SET_EVENT, this.visitor);
            }
            else if (visitorChanged) {
                this.publishEvent(this.VISITOR_CHANGE_EVENT, this.visitor);
            }
        }
    };
    RpiWebClient.prototype.parseVisitor = function (data, storeVisitorId) {
        if (!storeVisitorId) {
            storeVisitorId = true;
        }
        if (data) {
            var result = JSON.parse(data);
            if (storeVisitorId)
                this.setVisitor(result);
            if (result.ImpressionID != undefined)
                this.pageParameters.impressionId = result.ImpressionID;
            return result;
        }
        else {
            return null;
        }
    };
    RpiWebClient.prototype.getGeolocationCookie = function () {
        var value = "";
        var cookieValue = this.getCookie(this.config.clientId + "_g");
        if (cookieValue) {
            value = cookieValue;
        }
        return value;
    };
    RpiWebClient.prototype.setGeolocationCookie = function () {
        if (this.config.clientId) {
            var expiryDays = 5;
            if (this.config.geolocationCookieDuration) {
                expiryDays = this.config.geolocationCookieDuration;
            }
            this.setCookie(this.config.clientId + "_g", Date.now().toString(), expiryDays, this.config.realtimeCookieDomain);
        }
    };
    RpiWebClient.prototype.getApiServer = function () {
        if (this.config.serverUrl) {
            if (this.config.serverUrl.indexOf("http") == 0) {
                return this.config.serverUrl;
            }
            else {
                return (("https:" == document.location.protocol) ? "https://" + this.config.serverUrl : "http://" + this.config.serverUrl);
            }
        }
        return "";
    };
    RpiWebClient.prototype.getCacheVisitUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + "/api/Cache/Visit";
    };
    RpiWebClient.prototype.getCacheVisitorsAttributesUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + "/api/cache/visitors/attributes/";
    };
    RpiWebClient.prototype.getCacheCachedAttributesUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + "/api/Cache/CachedAttributes";
    };
    RpiWebClient.prototype.getEventsUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + '/api/Events';
    };
    RpiWebClient.prototype.getDecisionsUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + "/api/Decisions/";
    };
    RpiWebClient.prototype.getDecisionsContextsUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + "/api/Decisions/Contexts/";
    };
    RpiWebClient.prototype.getVisitorViewUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + "/api/Cache/Visitors/" + this.config.clientId + "/Views/";
    };
    RpiWebClient.prototype.getXhr = function () {
        var xmlHttp;
        try {
            xmlHttp = new XMLHttpRequest();
        }
        catch (e) {
            try {
                xmlHttp = new ActiveXObject("Msxml2.XMLHTTP");
            }
            catch (e) {
                xmlHttp = new ActiveXObject("Microsoft.XMLHTTP");
            }
        }
        if (!xmlHttp) {
            return false;
        }
        return xmlHttp;
    };
    RpiWebClient.prototype._renderDynamicContent = function (dynamicContents) {
        var isContextLookup = false;
        var masterMessageDivs = [];
        if (this.decisionOptions.isContextLookup) {
            isContextLookup = this.decisionOptions.isContextLookup;
        }
        if (this.decisionOptions.masterMessageDivs) {
            masterMessageDivs = this.decisionOptions.masterMessageDivs;
        }
        if (this.visitor) {
            if (isContextLookup || dynamicContents.length == 1) {
                for (var _i = 0, dynamicContents_1 = dynamicContents; _i < dynamicContents_1.length; _i++) {
                    var dynamicContent = dynamicContents_1[_i];
                    var isMasterMessage = (masterMessageDivs && masterMessageDivs.indexOf(dynamicContent.divName) >= 0);
                    this._renderDecision(dynamicContent, isMasterMessage);
                }
            }
            else {
                this._renderMultipleDecisions(dynamicContents);
            }
            return;
        }
        else {
            this.log("Unable to render dynamic content, visitor not set", "WARN");
        }
        if (dynamicContents && dynamicContents.length) {
            for (var _a = 0, dynamicContents_2 = dynamicContents; _a < dynamicContents_2.length; _a++) {
                var dynamicContent = dynamicContents_2[_a];
                var result = {
                    ContentID: dynamicContent.contentId,
                    ResultID: dynamicContent.defaultKey,
                    DefaultKey: dynamicContent.defaultKey,
                    DivName: dynamicContent.divName,
                    ContentPath: this.decisionOptions.defaultContentAddress + dynamicContent.contentId + "/" + dynamicContent.defaultKey + "/0.htm",
                    SlotName: "",
                    ContentName: "",
                    Result: "",
                    ResultContent: "",
                    ResultContentXML: "",
                    ContentFormat: "",
                    IsCachedContent: false,
                    IsEmptyResult: false,
                    PluginTag: "",
                    PluginTagDefault: "",
                    ContextName: "",
                    VisitorID: "",
                    DecisionType: ""
                };
                if (!dynamicContent.isCachedContent) {
                    this.renderDecisionContent(result);
                }
            }
        }
    };
    RpiWebClient.prototype._getDecision = function (dynamicContent, isMasterMessage, callback) {
        var isContextLookup = false;
        if (this.decisionOptions.isContextLookup) {
            isContextLookup = this.decisionOptions.isContextLookup;
        }
        if (this.visitor) {
            var deviceIDParam = "";
            if (this.visitor.deviceId)
                deviceIDParam = "&deviceID=" + this.visitor.deviceId;
            var impressionIDParam = "";
            if (this.pageParameters.impressionId)
                impressionIDParam = "&impressionID=" + this.pageParameters.impressionId;
            var url = "";
            if (isContextLookup) {
                url = this.getDecisionsContextsUrl() + this.config.clientId + "/" + dynamicContent.contentId + "/" +
                    this.visitor.visitorId + "?minuteOffSetUTC=" + encodeURIComponent(new Date().getTimezoneOffset().toString()) +
                    deviceIDParam + impressionIDParam;
            }
            else {
                var masterMessageParam = "&isMasterMessage=false";
                if (isMasterMessage) {
                    masterMessageParam = "&isMasterMessage=true";
                }
                url = this.getDecisionsUrl() + dynamicContent.contentId + "/" + this.visitor.visitorId +
                    "?publishID=" + this.pageParameters.pagePublishId + deviceIDParam + impressionIDParam + masterMessageParam +
                    "&minuteOffSetUTC=" + encodeURIComponent(new Date().getTimezoneOffset().toString());
            }
            this.requestAsync(url, function (data) {
                if (isContextLookup) {
                    var results = this.parseDecisionResponse(data);
                    if (results.length == 0) {
                        return;
                    }
                    callback.call(this, results);
                    return;
                }
                else {
                    var result = this.parseDecisionResponse(data);
                    if (result.IsEmptyResult || result.Result == "0") {
                        return;
                    }
                    callback.call(this, [result]);
                    return;
                }
            });
        }
        if (!isContextLookup && !dynamicContent.isCachedContent) {
            var result = {
                ContentID: dynamicContent.contentId,
                ResultID: dynamicContent.defaultKey,
                DefaultKey: dynamicContent.defaultKey,
                DivName: dynamicContent.divName,
                ContentPath: this.decisionOptions.defaultContentAddress + dynamicContent.contentId + "/" + dynamicContent.defaultKey + "/0.htm",
                SlotName: "",
                ContentName: "",
                Result: "",
                ResultContent: "",
                ResultContentXML: "",
                ContentFormat: "",
                IsCachedContent: false,
                IsEmptyResult: false,
                PluginTag: "",
                PluginTagDefault: "",
                ContextName: "",
                VisitorID: "",
                DecisionType: ""
            };
            callback.call(this, [result]);
        }
    };
    RpiWebClient.prototype._renderDecision = function (dynamicContent, isMasterMessage) {
        var lookupRequest = "";
        var isContextLookup = false;
        if (this.decisionOptions.lookupRequest) {
            lookupRequest = this.decisionOptions.lookupRequest;
        }
        if (this.decisionOptions.isContextLookup) {
            isContextLookup = this.decisionOptions.isContextLookup;
        }
        this._getDecision(dynamicContent, isMasterMessage, function (results) {
            for (var _i = 0, results_3 = results; _i < results_3.length; _i++) {
                var result = results_3[_i];
                var elementId = dynamicContent.divName;
                if (isContextLookup) {
                    elementId = result.SlotName;
                }
                this.renderDecisionContent(result, elementId, lookupRequest);
            }
        });
    };
    RpiWebClient.prototype._renderMultipleDecisions = function (dynamicContents) {
        var lookupRequest = "";
        var masterMessageDivs = [];
        if (this.decisionOptions.lookupRequest) {
            lookupRequest = this.decisionOptions.lookupRequest;
        }
        if (this.decisionOptions.masterMessageDivs) {
            masterMessageDivs = this.decisionOptions.masterMessageDivs;
        }
        if (this.visitor) {
            var contentIds = [];
            var divIds = [];
            var masterMsgs = [];
            var defaultKeys = [];
            for (var i = 0; i < dynamicContents.length; i++) {
                var dynamicContent = dynamicContents[i];
                contentIds.push(dynamicContent.contentId);
                divIds.push(dynamicContent.divName);
                masterMsgs.push((masterMessageDivs && masterMessageDivs.indexOf(dynamicContent.divName) >= 0).toString());
                defaultKeys.push(dynamicContent.defaultKey);
            }
            if (!contentIds.length)
                return;
            var deviceIdParam = "";
            if (this.visitor.deviceId) {
                deviceIdParam = "&deviceID=" + this.visitor.deviceId;
            }
            var impressionIdParam = "";
            if (this.pageParameters.impressionId)
                impressionIdParam = "&impressionID=" + this.pageParameters.impressionId;
            var url = this.getDecisionsUrl() + this.visitor.visitorId + "?publishID=" + this.pageParameters.pagePublishId +
                deviceIdParam + impressionIdParam + "&minuteOffSetUTC=" +
                encodeURIComponent(new Date().getTimezoneOffset().toString()) + "&contentIDs=" + contentIds.join("|") +
                "&divNames=" + divIds.join("|") + "&defaultKeys=" + defaultKeys.join("|") + "&IsMasterMessages=" + masterMsgs.join("|");
            this.requestAsync(url, function (data) {
                var results = this.parseDecisionResponse(data);
                for (var i = 0; i < results.length; i++) {
                    var result = results[i];
                    if (result.IsEmptyResult || result.Result == "0")
                        continue;
                    this.renderDecisionContent(result, null, lookupRequest);
                }
            });
            return;
        }
        if (dynamicContents && dynamicContents.length) {
            for (var _i = 0, dynamicContents_3 = dynamicContents; _i < dynamicContents_3.length; _i++) {
                var dynamicContent = dynamicContents_3[_i];
                var result = {
                    ContentID: dynamicContent.contentId,
                    ResultID: dynamicContent.defaultKey,
                    DefaultKey: dynamicContent.defaultKey,
                    DivName: dynamicContent.divName,
                    ContentPath: this.decisionOptions.defaultContentAddress + dynamicContent.contentId + "/" + dynamicContent.defaultKey + "/0.htm",
                    SlotName: "",
                    ContentName: "",
                    Result: "",
                    ResultContent: "",
                    ResultContentXML: "",
                    ContentFormat: "",
                    IsCachedContent: false,
                    IsEmptyResult: false,
                    PluginTag: "",
                    PluginTagDefault: "",
                    ContextName: "",
                    VisitorID: "",
                    DecisionType: ""
                };
                if (!dynamicContent.isCachedContent) {
                    this.renderDecisionContent(result);
                }
            }
        }
    };
    RpiWebClient.prototype._setFormTracking = function () {
        if (this.pageParameters.cId) {
            var hiddenElements_1 = document.getElementsByName('rpiCID');
            for (var i = 0; i < hiddenElements_1.length; i++) {
                hiddenElements_1[i].value = this.pageParameters.cId;
            }
        }
        if (this.pageParameters.exId > 0) {
            var hiddenElements = document.getElementsByName('rpiExID');
            for (var i = 0; i < hiddenElements.length; i++) {
                hiddenElements[i].value = this.pageParameters.exId.toString();
            }
        }
    };
    RpiWebClient.prototype._submitCacheDBData = function (lookupId, lookupValue, lookupOnName, callback) {
        if (this.visitor) {
            var content = {};
            content["VisitorID"] = this.visitor.visitorId;
            content["DeviceID"] = this.visitor.deviceId;
            if (lookupOnName) {
                content["CachedAttributeName"] = lookupId;
            }
            else {
                content["CachedAttributeName"] = "";
                content["CachedAttributeListID"] = lookupId;
            }
            content["LookupValue"] = lookupValue;
            this.postRequestAsync(this.getCacheCachedAttributesUrl(), JSON.stringify(content), function (data) {
                this.parseVisitor(data, true);
                if (callback) {
                    callback.call(this, data);
                }
            });
        }
    };
    RpiWebClient.prototype._cacheDBData = function (lookupId, lookupValue) {
        if (this.visitor != null) {
            this._submitCacheDBData(lookupId, lookupValue, false);
        }
        else {
            this.log("Unable to cache DB data, visitor not set", "WARN");
        }
    };
    RpiWebClient.prototype._pushPageGoal = function (goalName, stateName, goalDetail, contentIds) {
        var goal = {
            name: goalName,
            state: stateName,
            detail: goalDetail,
            content: contentIds
        };
        if (!this.pageParameters.goals) {
            this.pageParameters.goals = [];
        }
        this.pageParameters.goals.push(goal);
    };
    RpiWebClient.prototype._submitFBParams = function (FB, edges, reload) {
        var batchArray = [];
        batchArray[0] = { method: "GET", relative_url: "me" };
        for (var i = 0; i < edges.length; i++) {
            batchArray[i + 1] = { method: "GET", relative_url: "me/" + edges[i] };
        }
        var self = this;
        FB.api('/', 'POST', {
            batch: batchArray
        }, function (responses) {
            if (responses.length && responses[0].body) {
                var response_1 = JSON.parse(responses[0].body);
                self.pushRealtimeParameter("Facebook_ID", response_1.id);
                self.pushRealtimeParameter("Facebook_Name", response_1.name);
                self.pushRealtimeParameter("Facebook_FirstName", response_1.first_name);
                self.pushRealtimeParameter("Facebook_LastName", response_1.last_name);
                self.pushRealtimeParameter("Facebook_Is_Logged_In", "true");
                if (response_1.username) {
                    self.pushRealtimeParameter("Facebook_username", response_1.username);
                }
                if (response_1.gender) {
                    self.pushRealtimeParameter("Facebook_Gender", response_1.gender);
                }
                if (response_1.locale) {
                    self.pushRealtimeParameter("Facebook_Locale", response_1.locale);
                }
                if (response_1.email) {
                    self.pushRealtimeParameter("Facebook_Email", response_1.email);
                }
                if (response_1.birthday) {
                    self.pushRealtimeParameter("Facebook_Birthday", response_1.birthday);
                }
                if (response_1.location) {
                    self.pushRealtimeParameter("Facebook_Location", response_1.location.name);
                }
                if (response_1.hometown) {
                    self.pushRealtimeParameter("Facebook_Hometown", response_1.hometown.name);
                }
            }
            for (var i = 0; i < edges.length; i++) {
                if (responses.length >= i + 2 && responses[0].body && responses[i + 1].body) {
                    var response = JSON.parse(responses[i + 1].body);
                    if (response.data) {
                        var nameValues = '';
                        for (var j = 0; j < response.data.length; j++) {
                            if (response.data[j].name) {
                                if (nameValues != '') {
                                    nameValues = nameValues + '_';
                                }
                                nameValues = nameValues + response.data[j].name;
                            }
                        }
                        if (nameValues) {
                            if (edges[i] == 'likes') {
                                self.pushRealtimeParameter("Facebook_Likes", nameValues);
                            }
                            else if (edges[i] == 'events') {
                                self.pushRealtimeParameter("Facebook_Events", nameValues);
                            }
                            else {
                                self.pushRealtimeParameter("Facebook_" + edges[i], nameValues);
                            }
                        }
                    }
                }
            }
            if (!self.realtimeParameters.length) {
                self.pushRealtimeParameter("Facebook_Is_Logged_In", "false");
            }
            if (reload == true) {
                self.submitVisitorDetails(function () { location.reload(); });
            }
            else {
                self.submitVisitorDetails();
            }
        });
    };
    return RpiWebClient;
}());
var rpiWebClient = new RpiWebClient();
