"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var RpiWebClient = (function () {
    function RpiWebClient() {
        var _this = this;
        this.defaults = {};
        this.realtimeParameters = [];
        this.isInitialized = false;
        this.decisionResults = [];
        this.config = {};
        this.visitor = { profile: { VisitorID: "", IsMasterKey: false, HasAlternativeKey: false, Keys: [] }, geolocationExpiry: 0 };
        this.LOAD_EVENT = "rpiWebClientLoad";
        this.PRE_INIT_EVENT = "rpiWebClientPreInit";
        this.INIT_VISITOR_EVENT = "rpiWebClientInitVisitor";
        this.INIT_EVENT = "rpiWebClientInit";
        this.RENDER_CONTEXT_EVENT = "rpiWebClientContextDecisionRender";
        this.RENDER_CONTENT_EVENT = "rpiWebClientDecisionRender";
        this.RENDER_LOOKUP_EVENT = "rpiWebClientLookupRequestRender";
        this.VISITOR_SET_EVENT = "rpiWebClientVisitorSet";
        this.VISITOR_CHANGE_EVENT = "rpiWebClientVisitorChange";
        this.DECISION_RESULT_CLICK_EVENT = "rpiDecisionResultLinkClick";
        this.DECISION_RESULT_LOAD_EVENT = "rpiDecisionResultLoad";
        if (typeof rpiWebClientDefaults !== "undefined") {
            this.defaults = rpiWebClientDefaults;
        }
        this.decisionOptions = {
            numberOfResults: 0,
            applyNumberResultsToNested: true,
            minuteOffsetUTC: encodeURIComponent(new Date().getTimezoneOffset().toString())
        };
        this.pageParameters = {
            pageTitle: "",
            goals: [],
            pendingEvents: [],
            impressionId: "",
            cId: "",
            exId: 0,
            geolocation: null,
            pagePublishId: "0"
        };
        this.urlParams = this.serializeQueryString();
        this.parseRpiPageParams();
        var self = this;
        setTimeout(function () {
            self.publishEvent(_this.LOAD_EVENT, self);
        }, 0);
    }
    RpiWebClient.prototype.init = function (opts, callback) {
        var self = this;
        if (!opts) {
            opts = {};
        }
        this.config = __assign({}, this.defaults, opts);
        this.log("config: " + JSON.stringify(this.config), "INFO");
        this.publishEvent(this.PRE_INIT_EVENT, this);
        this.visitor = this.getVisitorCookie();
        this.log("visitor: " + JSON.stringify(this.visitor), "INFO");
        this.publishEvent(this.INIT_VISITOR_EVENT, this);
        this.registerForClickEvents();
        if (this.config.requestGeolocation) {
            this.requestGeolocation(function (position) {
                if (position.coords) {
                    this.pageParameters.geolocation = position;
                    this.flushRealtimeParameters();
                }
            });
        }
        this.subscribeEvent(window, this.VISITOR_CHANGE_EVENT, function (event) {
            self.log("Handling " + self.VISITOR_CHANGE_EVENT + " event... pushing PageVisit web event.");
            var pageTitle = document.title;
            if (self.config.enablePageVisitTracking) {
                self.pushWebEvent('PageVisit', null, pageTitle);
            }
        });
        this.submitVisitorDetails(function (data) {
            if (this.config.contexts) {
                this.renderContextDecisions(this.config.contexts);
            }
            if (this.config.contentDecisions) {
                this.renderDecisions(this.config.contentDecisions);
            }
            this.publishEvent(this.INIT_EVENT, this);
            this.isInitialized = true;
            if (callback) {
                callback();
            }
        });
    };
    RpiWebClient.prototype.log = function (message, logLevel) {
        if (!logLevel) {
            logLevel = "LOG";
        }
        if (this.config.debug) {
            switch (logLevel) {
                case "INFO":
                    console.info(message);
                    return;
                case "WARN":
                    console.warn(message);
                    return;
                case "ERROR":
                    console.error(message);
                    return;
            }
            console.log(message);
        }
    };
    ;
    RpiWebClient.prototype.subscribeEvent = function (element, type, callback) {
        try {
            element.addEventListener(type, callback, false);
        }
        catch (e) {
            element.attachEvent('on' + type, callback);
        }
    };
    ;
    RpiWebClient.prototype.unsubscribeEvent = function (element, type, callback) {
        if (element.removeEventListener) {
            element.removeEventListener(type, callback, false);
        }
        else if (element.detachEvent) {
            element.detachEvent("on" + type, callback);
        }
        else {
            element["on" + type] = null;
        }
    };
    RpiWebClient.prototype.publishEvent = function (type, detail) {
        if (document.createEvent) {
            var event_1;
            try {
                event_1 = new CustomEvent(type, { detail: detail });
            }
            catch (e) {
                event_1 = document.createEvent("CustomEvent");
                event_1.initCustomEvent(type, false, false, detail);
            }
            window.dispatchEvent(event_1);
        }
        else {
            var event_2 = document.createEventObject();
            event_2.detail = detail;
            window.fireEvent('on' + type, event_2);
        }
    };
    ;
    RpiWebClient.prototype.serializeQueryString = function (queryString) {
        if (!queryString) {
            queryString = location.search.slice(1);
        }
        if (!queryString.length) {
            return null;
        }
        var pairs = queryString.split('&');
        var result = {};
        for (var _i = 0, pairs_1 = pairs; _i < pairs_1.length; _i++) {
            var pair = pairs_1[_i];
            var parts = pair.split('=');
            var key = parts[0].toLowerCase();
            if (result[key]) {
                result[key] = decodeURIComponent(result[key]) + "," + decodeURIComponent(parts[1] || '');
            }
            else {
                result[decodeURIComponent(key)] = decodeURIComponent(parts[1] || '');
            }
        }
        return result;
    };
    ;
    RpiWebClient.prototype.setElementData = function (element, name, value) {
        if (element.dataset) {
            element.dataset[name] = value;
        }
        else {
            element.setAttribute("data-" + name, value);
        }
    };
    RpiWebClient.prototype.getElementData = function (element, name) {
        if (element.dataset) {
            var value = element.dataset[name];
            if (value) {
                return value;
            }
        }
        else {
            var value = element.getAttribute("data-" + name);
            if (value) {
                return value;
            }
        }
        return "";
    };
    RpiWebClient.prototype.setLocalStorageItem = function (key, value) {
        localStorage.setItem(key, JSON.stringify(value));
    };
    RpiWebClient.prototype.getLocalStorageItem = function (key) {
        var storageItem = localStorage.getItem(key);
        if (storageItem) {
            return JSON.parse(storageItem);
        }
        return null;
    };
    RpiWebClient.prototype.detectBrowser = function () {
        var browser = "OTHER";
        if (navigator.userAgent.indexOf("Edge") > -1) {
            return "EDGE";
        }
        else if (navigator.userAgent.indexOf("OPR") > -1) {
            return "OPERA";
        }
        else if (navigator.userAgent.indexOf("Chrome") > -1) {
            return "CHROME";
        }
        else if (navigator.userAgent.indexOf("Safari") > -1) {
            return "SAFARI";
        }
        else if (navigator.userAgent.indexOf("Firefox") > -1) {
            return "FIREFOX";
        }
        else if (navigator.userAgent.indexOf("MSIE") > -1 || navigator.userAgent.indexOf("Trident") > -1) {
            return "MSIE";
        }
        return browser;
    };
    RpiWebClient.prototype.getUuid = function () {
        var dt = new Date().getTime();
        var uuid = 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
            var r = (dt + Math.random() * 16) % 16 | 0;
            dt = Math.floor(dt / 16);
            return (c == 'x' ? r : (r & 0x3 | 0x8)).toString(16);
        });
        return uuid;
    };
    RpiWebClient.prototype.parseRpiPageParams = function () {
        if (this.urlParams) {
            if (this.urlParams["exid"]) {
                this.pageParameters.exId = Number(this.urlParams["exid"]);
            }
            if (this.urlParams["rpcid"]) {
                this.pageParameters.cId = this.urlParams["rpcid"];
            }
        }
    };
    RpiWebClient.prototype.submitVisitorDetails = function (callback) {
        this.log("Submitting visitor details...", "INFO");
        var pageTitle = document.title;
        if (this.pageParameters.pageTitle) {
            pageTitle = this.pageParameters.pageTitle;
        }
        this.submitVisitorRequestAsync(false, function (data) {
            if (callback) {
                callback.call(this, data);
            }
            if (this.config.enablePageVisitTracking) {
                this.pushWebEvent('PageVisit', null, pageTitle);
            }
            var storageItems = this.getLocalStorageItem("PageLinkClick");
            if (storageItems) {
                for (var _i = 0, storageItems_1 = storageItems; _i < storageItems_1.length; _i++) {
                    var item = storageItems_1[_i];
                    this.sendMetric("PageLinkClick", null, item.href, null, item.metadata);
                }
                this.setLocalStorageItem("PageLinkClick", null);
            }
        });
    };
    RpiWebClient.prototype.getContextDecision = function (context, callback, slotName, returnAll) {
        this.log("Getting decisions for context: " + context, "INFO");
        var slotNameParam = "";
        if (slotName)
            slotNameParam = "&slotName=" + slotName;
        if (returnAll == undefined) {
            returnAll = true;
        }
        var self = this;
        if (this.visitor.profile.VisitorID) {
            var url = this.getDecisionsContextsUrl() + context + "?returnAll=" + returnAll + slotNameParam;
            var decisionRequest = this.getApiDecisionRequest(false);
            this.postRequestAsync(url, JSON.stringify(decisionRequest), function (data) {
                var results = this.parseResponseJSON(data);
                this.cacheDecisionResults(results);
                callback.call(self, results);
            }, undefined, this.shouldUseXhrCredentials());
        }
        else {
            this.log("Unable to get context decision, visitor not set", "WARN");
        }
    };
    RpiWebClient.prototype.renderContextDecision = function (context) {
        this.log("Rendering content for context: " + context, "INFO");
        this.getContextDecision(context, function (results) {
            for (var _i = 0, results_1 = results; _i < results_1.length; _i++) {
                var result = results_1[_i];
                this.renderDecisionContent(result);
            }
            this.publishEvent(this.RENDER_CONTEXT_EVENT, results);
        });
    };
    RpiWebClient.prototype.renderContextDecisions = function (contexts) {
        this.log("Rendering content for contexts: " + JSON.stringify(contexts), "INFO");
        for (var i = 0; i < contexts.length; i++) {
            this.renderContextDecision(contexts[i]);
        }
    };
    RpiWebClient.prototype.getDecision = function (contentId, isMasterMessage, callback) {
        this.log("Getting decision for content id: " + contentId, "INFO");
        var self = this;
        if (this.visitor.profile.VisitorID) {
            var url = this.getDecisionsUrl() + contentId;
            var decisionRequest = this.getApiDecisionRequest(isMasterMessage);
            this.postRequestAsync(url, JSON.stringify(decisionRequest), function (data) {
                var results = this.parseResponseJSON(data);
                this.cacheDecisionResults(results);
                callback.call(self, results);
            }, undefined, this.shouldUseXhrCredentials());
        }
        else {
            this.log("Unable to get decision, visitor not set", "WARN");
        }
    };
    RpiWebClient.prototype.renderDecision = function (contentId, elementId) {
        this.log("Rendering decision for content id: " + contentId, "INFO");
        var isMasterMessage = this.getIsMasterMessage(elementId);
        this.getDecision(contentId, isMasterMessage, function (results) {
            for (var _i = 0, results_2 = results; _i < results_2.length; _i++) {
                var result = results_2[_i];
                this.renderDecisionContent(result, elementId);
                this.publishEvent(this.RENDER_CONTENT_EVENT, result);
            }
        });
    };
    RpiWebClient.prototype.renderDecisions = function (contentDecisions) {
        this.log("Rendering content decisions: " + JSON.stringify(contentDecisions), "INFO");
        for (var _i = 0, contentDecisions_1 = contentDecisions; _i < contentDecisions_1.length; _i++) {
            var contentDecision = contentDecisions_1[_i];
            this.renderDecision(contentDecision.contentId, contentDecision.elementId);
        }
    };
    RpiWebClient.prototype.cacheDecisionResults = function (results) {
        if (this.config.enableDecisionCaching) {
            this.decisionResults = this.decisionResults.concat(results);
        }
        this.publishEvent(this.DECISION_RESULT_LOAD_EVENT, results);
    };
    RpiWebClient.prototype.getCachedVisitor = function (callback) {
        rpiWebClient.log("Getting cached visitor...");
        var url = this.getCacheParametersUrl(true) + "?visitorID=" + this.visitor.profile.VisitorID;
        this.requestAsync(url, function (data) {
            this.parseVisitor(data, true);
            if (callback) {
                callback.call(this, data);
            }
        }, undefined, this.shouldUseXhrCredentials());
    };
    RpiWebClient.prototype.getVisitorView = function (viewName, callback) {
        this.log("Getting visitor view: " + viewName, "INFO");
        var self = this;
        if (this.visitor.profile.VisitorID) {
            var url = this.getVisitorViewUrl() + viewName;
            var visitorViewArguments = {
                Identity: { VisitorID: this.visitor.profile.VisitorID },
                Attributes: this.getRealtimeParameters()
            };
            this.postRequestAsync(url, JSON.stringify(visitorViewArguments), function (data) {
                callback.call(this, this.parseResponseJSON(data));
            }, undefined, this.shouldUseXhrCredentials());
        }
        else {
            this.log("Unable to get visitor view, visitor not set", "WARN");
        }
    };
    RpiWebClient.prototype.getLookupAttributes = function (lookupRequest, callback) {
        this.log("Getting lookup attributes for request: " + lookupRequest, "INFO");
        if (this.visitor.profile.VisitorID) {
            var url = this.getCacheVisitorsAttributesUrl() + this.visitor.profile.VisitorID;
            this.postRequestAsync(url, lookupRequest, function (data) {
                callback.call(this, this.parseResponseJSON(data));
            }, undefined, this.shouldUseXhrCredentials());
        }
        else {
            this.log("Unable to get lookup attributes, visitor not set", "WARN");
        }
    };
    RpiWebClient.prototype.renderLookupAttributes = function (lookupRequest) {
        this.log("Rendering lookup attributes for request: " + lookupRequest, "INFO");
        this.getLookupAttributes(lookupRequest, function (results) {
            for (var _i = 0, results_3 = results; _i < results_3.length; _i++) {
                var result = results_3[_i];
                var elements = document.getElementsByName(result.ID);
                for (var i = 0; i < elements.length; i++) {
                    elements[i].innerHTML = result.Value;
                }
            }
            this.publishEvent(this.RENDER_LOOKUP_EVENT, results);
        });
    };
    RpiWebClient.prototype.submitVisitorRequestAsync = function (updateProfileOnly, onSuccess, onError) {
        var url = this.getCacheVisitUrl() + "?updateProfileOnly=" + updateProfileOnly;
        var regDetails = this.getVisitorRegistrationDetails();
        this.postRequestAsync(url, JSON.stringify(regDetails), function (data) {
            this.parseVisitor(data, true);
            if (onSuccess) {
                onSuccess.call(this, data);
            }
        }, onError, this.shouldUseXhrCredentials());
    };
    RpiWebClient.prototype.parseResponseJSON = function (data) {
        this.log("Parsing response data: " + data, "INFO");
        var responseJSON = JSON.parse(data);
        return responseJSON;
    };
    RpiWebClient.prototype.registerForClickEvents = function (enableLinkTracking) {
        var self = this;
        if (enableLinkTracking == undefined) {
            if (this.config.enableClickTracking == undefined) {
                enableLinkTracking = false;
            }
            else {
                enableLinkTracking = this.config.enableClickTracking;
            }
        }
        if (!enableLinkTracking) {
            return;
        }
        var elements = document.getElementsByTagName('a');
        for (var i = 0, len = elements.length; i < len; i++) {
            this.subscribeEvent(elements[i], "click", this.webEventClickHandler);
        }
    };
    RpiWebClient.prototype.webEventClickHandler = function (event) {
        if (event.target) {
            var element = event.target;
            while (element.nodeName != "A") {
                element = element.parentElement;
                if (!element) {
                    element = event.target;
                    break;
                }
            }
            if (element && element.nodeName == "A") {
                var anchor = element;
                if (anchor.href) {
                    var metadata = [];
                    var resultId = rpiWebClient.getElementData(element, "resultId");
                    if (resultId) {
                        rpiWebClient.publishEvent(rpiWebClient.DECISION_RESULT_CLICK_EVENT, resultId);
                        metadata = [{ Name: "resultId", Value: resultId }];
                    }
                    rpiWebClient.sendLinkClickEvent(anchor.href, metadata);
                }
            }
        }
    };
    RpiWebClient.prototype.getRealtimeParameters = function () {
        var exclList = ["rpcid", "exid", "rpimg"];
        var params = [];
        if (this.urlParams) {
            if (this.config.enableUrlParameterCaching) {
                for (var key in this.urlParams) {
                    var value = this.urlParams[key];
                    if (exclList.indexOf(key) < 0) {
                        params.push({ Name: key, Value: value });
                    }
                }
            }
            else {
                if (this.config.urlParameterCachingOverrides && this.config.urlParameterCachingOverrides.length > 0) {
                    for (var _i = 0, _a = this.config.urlParameterCachingOverrides; _i < _a.length; _i++) {
                        var param = _a[_i];
                        var key = param.toLocaleLowerCase();
                        if (this.urlParams[key]) {
                            params.push({ Name: key, Value: this.urlParams[key] });
                        }
                    }
                }
            }
        }
        if (this.realtimeParameters.length > 0) {
            params = params.concat(this.realtimeParameters);
            this.realtimeParameters = [];
        }
        return params;
    };
    RpiWebClient.prototype.getVisitorRegistrationDetails = function () {
        var publishId = "0";
        if (this.pageParameters.pagePublishId) {
            publishId = this.pageParameters.pagePublishId;
        }
        var exID = null;
        var cID = null;
        var msgIDs = null;
        if (this.urlParams) {
            exID = this.urlParams["exid"];
            cID = this.urlParams["rpcid"];
            msgIDs = this.urlParams["rpimg"];
        }
        var visitDetails = {};
        visitDetails["PagePublishedID"] = publishId;
        if (!this.visitor.profile.VisitorID) {
            visitDetails["IsNewVisitor"] = true;
        }
        else {
            visitDetails["IsNewVisitor"] = false;
            visitDetails["VisitorID"] = this.visitor.profile.VisitorID;
            visitDetails["DeviceID"] = this.visitor.profile.DeviceID;
        }
        var trackingObjects = {};
        if (exID) {
            trackingObjects["ChannelExecutionID"] = exID;
            if (cID) {
                trackingObjects["RPContactID"] = cID;
            }
        }
        if (msgIDs) {
            var msgComp = msgIDs.split("_");
            if (msgComp.length > 1) {
                trackingObjects["MessageListID"] = msgComp[0];
                trackingObjects["MessageID"] = msgComp[1];
            }
        }
        visitDetails["InteractionTracking"] = trackingObjects;
        if (this.config.clientId) {
            visitDetails["ClientID"] = this.config.clientId;
        }
        var referrer = document.referrer;
        if (referrer) {
            visitDetails["PageReferrer"] = referrer;
        }
        if (this.pageParameters.geolocation) {
            visitDetails["Geolocation"] = {
                "Longitude": this.pageParameters.geolocation.coords.longitude,
                "Latitude": this.pageParameters.geolocation.coords.latitude
            };
        }
        visitDetails["VisitorAttributes"] = this.getRealtimeParameters();
        return visitDetails;
    };
    RpiWebClient.prototype.getApiDecisionRequest = function (isMasterMessage) {
        var request = {};
        var deviceId = null;
        if (this.visitor.profile.VisitorID) {
            var identity = {
                VisitorID: this.visitor.profile.VisitorID
            };
            request.Identity = identity;
            if (this.visitor.profile.DeviceID) {
                deviceId = this.visitor.profile.DeviceID;
            }
        }
        var requestDetail = {};
        requestDetail.NumberOfResults = this.decisionOptions.numberOfResults;
        requestDetail.ApplyNumberResultsToNested = this.decisionOptions.applyNumberResultsToNested;
        requestDetail.ImpressionID = this.pageParameters.impressionId;
        requestDetail.PublishID = this.pageParameters.pagePublishId;
        requestDetail.MinuteOffsetUTC = this.decisionOptions.minuteOffsetUTC;
        requestDetail.APIContext = this.decisionOptions.apiContext;
        requestDetail.IsMasterMessage = isMasterMessage;
        if (deviceId) {
            requestDetail.DeviceID = deviceId;
        }
        if (this.realtimeParameters && this.realtimeParameters.length) {
            requestDetail.Attributes = this.realtimeParameters;
        }
        if (this.pageParameters.geolocation) {
            requestDetail.Geolocation = {
                "Longitude": this.pageParameters.geolocation.coords.longitude,
                "Latitude": this.pageParameters.geolocation.coords.latitude
            };
        }
        request.DecisionRequest = requestDetail;
        return request;
    };
    RpiWebClient.prototype.requestGeolocation = function (onSuccess) {
        if (!this.visitor.geolocationExpiry || this.visitor.geolocationExpiry < Date.now()) {
            this.setGeolocationExpiry();
            if (navigator.geolocation) {
                var self_1 = this;
                navigator.geolocation.getCurrentPosition(function (position) {
                    self_1.log("Geolocation request success: longitude: " + position.coords.longitude + " latitude: " + position.coords.latitude, "INFO");
                    onSuccess.call(self_1, position);
                }, function (error) {
                    self_1.log("Geolocation request failed with error code: " + error.code + ", message: " + error.message, "WARN");
                }, { maximumAge: 300000, timeout: 10000 });
            }
            else {
                this.log("Geolocation not supported by this browser", "WARN");
            }
        }
    };
    RpiWebClient.prototype.pushWebEvent = function (state, stateValue, extraParam, contentId, metadata) {
        if (!extraParam) {
            extraParam = null;
        }
        if (!contentId) {
            contentId = null;
        }
        if (!this.visitor.profile.VisitorID) {
            var rpiEvent = { state: state, stateValue: stateValue, extraParam: extraParam, contentID: contentId, metadata: metadata };
            this.pageParameters.pendingEvents.push(rpiEvent);
        }
        else {
            this.sendMetric(state, stateValue, extraParam, contentId, metadata);
            this.flushPendingEvents();
        }
    };
    RpiWebClient.prototype.flushPendingEvents = function () {
        if (this.visitor.profile.VisitorID) {
            var hiddenElements = document.getElementsByName('rpiVisitorID');
            for (var i = 0; i < hiddenElements.length; i++) {
                hiddenElements[i].value = this.visitor.profile.VisitorID;
            }
            if (this.visitor.profile.DeviceID) {
                hiddenElements = document.getElementsByName('rpiDeviceID');
                for (var i = 0; i < hiddenElements.length; i++) {
                    hiddenElements[i].value = this.visitor.profile.DeviceID;
                }
            }
        }
        if (this.pageParameters.impressionId) {
            var hiddenElements = document.getElementsByName('rpiImpID');
            for (var i = 0; i < hiddenElements.length; i++) {
                hiddenElements[i].value = this.pageParameters.impressionId;
            }
        }
        if (this.pageParameters.pendingEvents) {
            for (var i = 0; i < this.pageParameters.pendingEvents.length; i++) {
                var event_3 = this.pageParameters.pendingEvents[i];
                this.sendMetric(event_3["state"], event_3["stateValue"], event_3["extraParam"], event_3["contentID"], event_3["metadata"]);
            }
            this.pageParameters.pendingEvents = [];
        }
    };
    RpiWebClient.prototype.sendMetric = function (state, stateValue, extraParam, contentID, metadata, callback) {
        var url = this.getEventsUrl();
        var eventDetails = this.getEventRequestDetails(state, stateValue, extraParam, contentID, metadata);
        this.postRequestAsync(url, JSON.stringify(eventDetails), function (data) {
            if (callback) {
                callback.call(this, data);
            }
        }, undefined, this.shouldUseXhrCredentials());
    };
    RpiWebClient.prototype.sendLinkClickEvent = function (href, metadata) {
        var url = this.getEventsUrl();
        var browser = this.detectBrowser();
        var eventDetails = this.getEventRequestDetails("PageLinkClick", null, href, null, metadata);
        if (navigator.sendBeacon) {
            navigator.sendBeacon(url, JSON.stringify(eventDetails));
        }
        else if (browser != "FIREFOX" && browser != "SAFARI" && browser != "OTHER") {
            this.postRequestAsync(url, JSON.stringify(eventDetails), undefined, undefined, this.shouldUseXhrCredentials());
        }
        else {
            var storageItem = this.getLocalStorageItem("PageLinkClick");
            if (!storageItem) {
                storageItem = [];
            }
            storageItem.push({ href: href, metadata: metadata });
            this.setLocalStorageItem("PageLinkClick", storageItem);
        }
    };
    RpiWebClient.prototype.getEventRequestDetails = function (state, stateValue, extraParam, contentID, metadata) {
        var eventDetails = {};
        eventDetails["EventName"] = state;
        if (this.pageParameters.exId > 0) {
            eventDetails["ChannelExecutionID"] = this.pageParameters.exId;
        }
        if (this.pageParameters.cId) {
            eventDetails["RPContactID"] = this.pageParameters.cId;
        }
        if (stateValue) {
            eventDetails["MetricValue"] = stateValue;
        }
        if (extraParam) {
            eventDetails["EventDetail"] = extraParam;
        }
        var publishId = "0";
        if (this.pageParameters.pagePublishId) {
            publishId = this.pageParameters.pagePublishId;
        }
        eventDetails["PagePublishedID"] = publishId;
        if (this.visitor.profile.VisitorID) {
            eventDetails["VisitorID"] = this.visitor.profile.VisitorID;
            eventDetails["DeviceID"] = this.visitor.profile.DeviceID;
        }
        if (this.pageParameters.impressionId)
            eventDetails["ImpressionID"] = this.pageParameters.impressionId;
        if (this.config.clientId) {
            eventDetails["ClientID"] = this.config.clientId;
        }
        if (contentID) {
            eventDetails["ContentID"] = contentID;
        }
        if (state == 'PageVisit') {
            if (this.pageParameters.exId > 0) {
                eventDetails["PageReferral"] = "exid" + this.pageParameters.exId;
            }
            else {
                var referrer = document.referrer;
                if (referrer) {
                    eventDetails["PageReferral"] = referrer;
                }
                else {
                    eventDetails["PageReferral"] = "Direct";
                }
            }
        }
        if (this.pageParameters.goals && this.pageParameters.goals.length > 0) {
            var goalDetail = {};
            for (var _i = 0, _a = this.pageParameters.goals; _i < _a.length; _i++) {
                var goal = _a[_i];
                if (goal["state"] && goal["state"].toLowerCase() == state.toLowerCase()) {
                    if (goal["detail"]) {
                        goalDetail["GoalContentID"] = goal["content"];
                        if (goal["name"]) {
                            goalDetail["GoalName"] = goal["name"];
                        }
                        eventDetails["GoalDetails"] = goalDetail;
                    }
                    else if (extraParam && extraParam.toLowerCase().indexOf(goal["detail"].toLowerCase()) >= 0) {
                        goalDetail["GoalContentID"] = goal["content"];
                        if (goal["name"]) {
                            goalDetail["GoalName"] = goal["name"];
                        }
                        eventDetails["GoalDetails"] = goalDetail;
                    }
                }
            }
        }
        if (metadata && metadata.length > 0) {
            eventDetails["Metadata"] = metadata;
        }
        return eventDetails;
    };
    RpiWebClient.prototype.renderDecisionContent = function (result, elementId, lookupRequest) {
        if (!elementId) {
            elementId = this.getElementId(result);
        }
        var element = this.getDomElement(elementId);
        if (!element) {
            this.log("Unable to find DOM element with ID: " + elementId, "WARN");
            return;
        }
        var decisionMeta = {
            ContentName: result.ContentName,
            ResultID: result.ResultID,
            ContentID: result.ContentID,
            Result: result.Result,
            ContentFormat: result.ContentFormat,
            DefaultKey: result.DefaultKey,
            DivName: result.DivName,
            IsEmptyResult: result.IsEmptyResult,
            PluginTag: result.PluginTag,
            PluginTagDefault: result.PluginTagDefault,
            ContextName: result.ContextName,
            SlotName: result.SlotName,
            DecisionType: result.DecisionType
        };
        element.setAttribute("rg-decision-meta", JSON.stringify(decisionMeta));
        this.getDecisionContent(result, function (content) {
            if (element) {
                if (result.ContentFormat == "HTML") {
                    element.innerHTML = "";
                    var elements = this.parseHtml(content);
                    for (var i = 0; i < elements.length; i++) {
                        element.appendChild(elements[i]);
                    }
                }
                else {
                    element.innerHTML = content;
                }
                var enableLinkTracking = false;
                if (this.config.enableClickTracking != undefined) {
                    enableLinkTracking = this.config.enableClickTracking;
                }
                if (result.ResultID) {
                    var anchors = element.getElementsByTagName("a");
                    for (var i = 0; i < anchors.length; i++) {
                        var anchor = anchors[i];
                        this.setElementData(anchor, "resultId", result.ResultID);
                        if (enableLinkTracking) {
                            this.subscribeEvent(anchor, "click", this.webEventClickHandler);
                        }
                    }
                }
            }
        }, lookupRequest);
    };
    RpiWebClient.prototype.getElementId = function (result) {
        if (result.SlotName) {
            return result.SlotName;
        }
        return result.DivName;
    };
    RpiWebClient.prototype.getDomElement = function (elementId) {
        var nodes = document.querySelectorAll("[rg-content-slot='" + elementId + "']");
        if (nodes && nodes.length) {
            return nodes[0];
        }
        return document.getElementById(elementId);
    };
    RpiWebClient.prototype.parseHtml = function (content) {
        var elements = [];
        var body = document.createElement("body");
        body.innerHTML = content;
        for (var i = 0; i < body.children.length; i++) {
            var child = body.children[i];
            if (child.nodeName == "SCRIPT") {
                var script = document.createElement("script");
                for (var i_1 in child.attributes) {
                    var attrib = child.attributes[i_1];
                    script.setAttribute(attrib.name, attrib.value);
                }
                script.innerHTML = child.innerHTML;
                elements.push(script);
            }
            else {
                elements.push(child);
            }
        }
        return elements;
    };
    RpiWebClient.prototype.getDecisionContent = function (result, callback, lookupRequest) {
        if (result.IsEmptyResult || result.Result == "0") {
            return;
        }
        var self = this;
        if (result.IsCachedContent) {
            callback.call(self, result.ResultContent);
            return;
        }
        this.requestAsync(result.ContentPath, function (data) {
            callback.call(this, data);
            if (lookupRequest && data.indexOf("-RPIEMBA-") >= 0) {
                this.renderLookupAttributes(lookupRequest);
            }
        }, undefined, this.shouldUseXhrCredentials());
    };
    RpiWebClient.prototype.pushRealtimeParameter = function (name, value) {
        this.log("Pushing name: " + name + ", value: " + value + " to realtimeParameters...", "INFO");
        this.realtimeParameters.push({ Name: name, Value: value });
    };
    RpiWebClient.prototype.flushRealtimeParameters = function (callback) {
        this.log("Pushing realtimeParameters to visitor profile: " + JSON.stringify(this.realtimeParameters), "INFO");
        if (this.pageParameters.geolocation) {
            this.submitVisitorRequestAsync(true, callback);
        }
        else if (this.realtimeParameters && this.realtimeParameters.length) {
            var url = this.getCacheParametersUrl();
            this.postRequestAsync(url, JSON.stringify(this.realtimeParameters), function (data) {
                this.parseVisitor(data, true);
                if (callback) {
                    callback.call(this, data);
                }
            }, undefined, this.shouldUseXhrCredentials());
            this.realtimeParameters = [];
        }
    };
    RpiWebClient.prototype.getIsMasterMessage = function (elementId) {
        var masterMessageDivs = [];
        if (this.decisionOptions.masterMessageDivs) {
            masterMessageDivs = this.decisionOptions.masterMessageDivs;
        }
        var isMasterMessage = (masterMessageDivs.indexOf(elementId) >= 0);
        return isMasterMessage;
    };
    RpiWebClient.prototype.requestAsync = function (url, onSuccess, onError, withCredentials) {
        this.log("Performing async GET request to " + url, "INFO");
        if (!withCredentials) {
            withCredentials = false;
        }
        var xmlHttp = this.getXhr(withCredentials);
        var self = this;
        xmlHttp.onreadystatechange = function () {
            if (this.readyState === 4) {
                if (this.status === 200 && typeof onSuccess == 'function') {
                    onSuccess.call(self, this.responseText);
                }
                else if (typeof onError == 'function') {
                    self.log("Error performing async GET:" + this.statusText, "ERROR");
                    onError.call(self, this.responseText);
                }
            }
        };
        xmlHttp.open("GET", url, true);
        xmlHttp.send();
    };
    RpiWebClient.prototype.postRequestAsync = function (url, contents, onSuccess, onError, withCredentials) {
        this.log("Performing async POST request to: " + url + " with contents: " + contents, "INFO");
        if (!withCredentials) {
            withCredentials = false;
        }
        var xmlHttp = this.getXhr(withCredentials);
        var self = this;
        xmlHttp.onreadystatechange = function () {
            if (this.readyState === 4) {
                if (this.status === 200 || this.status === 204) {
                    if (onSuccess) {
                        onSuccess.call(self, this.responseText);
                    }
                }
                else if (onError) {
                    self.log("Error performing async POST:" + this.statusText, "ERROR");
                    onError.call(self, this.responseText);
                }
            }
        };
        xmlHttp.open("POST", url, true);
        xmlHttp.setRequestHeader("Accept", "application/json");
        xmlHttp.setRequestHeader("Content-type", "text/plain");
        xmlHttp.send(contents);
    };
    RpiWebClient.prototype.getCookie = function (cookieName) {
        var cookies = document.cookie.split(";");
        for (var _i = 0, cookies_1 = cookies; _i < cookies_1.length; _i++) {
            var cookie = cookies_1[_i];
            var name_1 = cookie.substr(0, cookie.indexOf("="));
            var value = cookie.substr(cookie.indexOf("=") + 1);
            name_1 = name_1.replace(/^\s+|\s+$/g, "");
            if (name_1 == cookieName) {
                return unescape(value);
            }
        }
        return null;
    };
    RpiWebClient.prototype.setCookie = function (cookieName, cookieValue, days, domain, sameSite, secure, expiry) {
        if (!expiry) {
            expiry = new Date();
            if (days) {
                expiry.setDate(expiry.getDate() + days);
            }
        }
        var value = cookieName + "=" + escape(cookieValue)
            + ((!expiry) ? "" : "; Expires=" + expiry.toUTCString())
            + ((!domain) ? "" : "; Domain=" + domain)
            + ((!sameSite) ? "; SameSite=Lax" : "; SameSite=" + sameSite)
            + ((!secure) ? "" : "; Secure")
            + "; Path=/";
        this.log("setting cookie: " + value, "INFO");
        document.cookie = value;
    };
    RpiWebClient.prototype.getVisitorCookie = function () {
        var visitorCookie = { profile: { VisitorID: "", IsMasterKey: false, HasAlternativeKey: false, Keys: [] }, geolocationExpiry: 0 };
        var cookie = this.getCookie(this.config.clientId);
        if (!cookie) {
            visitorCookie.profile.VisitorID = this.getUuid();
            return visitorCookie;
        }
        else {
            try {
                var json = JSON.parse(cookie);
                if (typeof (json) == "object") {
                    if (json.profile) {
                        return json;
                    }
                    else {
                        this.log("Expected VisitorCookie, got: " + cookie, "ERROR");
                    }
                }
                else {
                    this.log("Expected VisitorCookie of type object, got: " + typeof (json), "WARN");
                }
            }
            catch (e) {
                this.log("Converting cookie for visitor ID: " + cookie, "LOG");
            }
        }
        var visitorId = cookie;
        var deviceId = "";
        var geolocationExpiry = 0;
        var details = this.getCookie(this.config.clientId + "_d");
        if (details) {
            deviceId = details;
        }
        details = this.getCookie(this.config.clientId + "_g");
        if (details) {
            geolocationExpiry = parseInt(details);
        }
        visitorCookie.profile.VisitorID = visitorId;
        visitorCookie.profile.DeviceID = deviceId;
        visitorCookie.geolocationExpiry = geolocationExpiry;
        return visitorCookie;
    };
    RpiWebClient.prototype.setVisitorCookie = function (apiVisitor) {
        if (this.config.clientId) {
            var visitorSet = false;
            var visitorChanged = false;
            if (!apiVisitor) {
                apiVisitor = this.visitor.profile;
            }
            if (!this.visitor.profile.VisitorID) {
                visitorSet = true;
            }
            else if (this.visitor.profile.VisitorID != apiVisitor.VisitorID) {
                visitorChanged = true;
            }
            this.visitor.profile = apiVisitor;
            var days = this.config.visitorCookieDuration;
            if (!days) {
                days = 365;
            }
            this.setCookie(this.config.clientId, JSON.stringify(this.visitor), days, this.config.realtimeCookieDomain, this.config.realtimeCookieSameSite, true);
            if (visitorSet) {
                this.publishEvent(this.VISITOR_SET_EVENT, this.visitor);
            }
            else if (visitorChanged) {
                this.publishEvent(this.VISITOR_CHANGE_EVENT, this.visitor);
            }
        }
    };
    RpiWebClient.prototype.parseVisitor = function (data, storeVisitorId) {
        if (storeVisitorId == undefined) {
            storeVisitorId = true;
        }
        if (data) {
            var result = JSON.parse(data);
            if (storeVisitorId)
                this.setVisitorCookie(result);
            if (result.ImpressionID != undefined)
                this.pageParameters.impressionId = result.ImpressionID;
            return result;
        }
        else {
            return null;
        }
    };
    RpiWebClient.prototype.setGeolocationExpiry = function () {
        if (this.config.clientId) {
            var expiryDays = 5;
            if (this.config.geolocationCookieDuration) {
                expiryDays = this.config.geolocationCookieDuration;
            }
            var today = new Date();
            var expiry = new Date(today.toString());
            expiry.setDate(expiry.getDate() + expiryDays);
            this.visitor.geolocationExpiry = expiry.valueOf();
            this.setVisitorCookie();
        }
    };
    RpiWebClient.prototype.getApiServer = function () {
        if (this.config.serverUrl) {
            if (this.config.serverUrl.indexOf("http") == 0) {
                return this.config.serverUrl;
            }
            else {
                return (("https:" == document.location.protocol) ? "https://" + this.config.serverUrl : "http://" + this.config.serverUrl);
            }
        }
        return "";
    };
    RpiWebClient.prototype.getCacheVisitUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + "/api/Cache/Visit";
    };
    RpiWebClient.prototype.getCacheParametersUrl = function (isGetRequest) {
        var serverUrl = this.getApiServer();
        if (isGetRequest) {
            return serverUrl + "/api/Cache/" + this.config.clientId + "/Parameters/";
        }
        return serverUrl + "/api/Cache/" + this.config.clientId + "/" + this.visitor.profile.VisitorID + "/Parameters/";
    };
    RpiWebClient.prototype.getCacheVisitorsAttributesUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + "/api/cache/visitors/attributes/";
    };
    RpiWebClient.prototype.getCacheCachedAttributesUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + "/api/Cache/CachedAttributes";
    };
    RpiWebClient.prototype.getEventsUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + '/api/Events';
    };
    RpiWebClient.prototype.getDecisionsUrl = function () {
        var serverUrl = this.getApiServer();
        if (this.config.clientId) {
            return serverUrl + "/api/Decisions/" + this.config.clientId + "/";
        }
        return serverUrl + "/api/Decisions/";
    };
    RpiWebClient.prototype.getDecisionsContextsUrl = function () {
        var serverUrl = this.getApiServer();
        if (this.config.clientId) {
            return serverUrl + "/api/Decisions/Contexts/" + this.config.clientId + "/";
        }
        return serverUrl + "/api/Decisions/Contexts/";
    };
    RpiWebClient.prototype.getVisitorViewUrl = function () {
        var serverUrl = this.getApiServer();
        return serverUrl + "/api/Cache/Visitors/" + this.config.clientId + "/Views/";
    };
    RpiWebClient.prototype.shouldUseXhrCredentials = function () {
        if (!this.config.serverCookieEnabled) {
            return false;
        }
        return true;
    };
    RpiWebClient.prototype.getXhr = function (withCredentials) {
        var xmlHttp;
        try {
            xmlHttp = new XMLHttpRequest();
        }
        catch (e) {
            try {
                xmlHttp = new ActiveXObject("Msxml2.XMLHTTP");
            }
            catch (e) {
                xmlHttp = new ActiveXObject("Microsoft.XMLHTTP");
            }
        }
        if (!xmlHttp) {
            return false;
        }
        xmlHttp.withCredentials = withCredentials;
        return xmlHttp;
    };
    RpiWebClient.prototype._setFormTracking = function () {
        if (this.pageParameters.cId) {
            var hiddenElements = document.getElementsByName('rpiCID');
            for (var i = 0; i < hiddenElements.length; i++) {
                hiddenElements[i].value = this.pageParameters.cId;
            }
        }
        if (this.pageParameters.exId > 0) {
            var hiddenElements = document.getElementsByName('rpiExID');
            for (var i = 0; i < hiddenElements.length; i++) {
                hiddenElements[i].value = this.pageParameters.exId.toString();
            }
        }
    };
    RpiWebClient.prototype._submitCacheDBData = function (lookupId, lookupValue, lookupOnName, callback) {
        if (this.visitor.profile.VisitorID) {
            var content = {};
            content["VisitorID"] = this.visitor.profile.VisitorID;
            if (this.visitor.profile.DeviceID) {
                content["DeviceID"] = this.visitor.profile.DeviceID;
            }
            if (lookupOnName) {
                content["CachedAttributeName"] = lookupId;
            }
            else {
                content["CachedAttributeName"] = "";
                content["CachedAttributeListID"] = lookupId;
            }
            content["LookupValue"] = lookupValue;
            this.postRequestAsync(this.getCacheCachedAttributesUrl(), JSON.stringify(content), function (data) {
                this.parseVisitor(data, true);
                if (callback) {
                    callback.call(this, data);
                }
            }, undefined, this.shouldUseXhrCredentials());
        }
    };
    RpiWebClient.prototype._cacheDBData = function (lookupId, lookupValue) {
        if (this.visitor.profile.VisitorID) {
            this._submitCacheDBData(lookupId, lookupValue, false);
        }
        else {
            this.log("Unable to cache DB data, visitor not set", "WARN");
        }
    };
    RpiWebClient.prototype._pushPageGoal = function (goalName, stateName, goalDetail, contentIds) {
        var goal = {
            name: goalName,
            state: stateName,
            detail: goalDetail,
            content: contentIds
        };
        if (!this.pageParameters.goals) {
            this.pageParameters.goals = [];
        }
        this.pageParameters.goals.push(goal);
    };
    RpiWebClient.prototype._submitFBParams = function (FB, edges, reload) {
        var batchArray = [];
        batchArray[0] = { method: "GET", relative_url: "me" };
        for (var i = 0; i < edges.length; i++) {
            batchArray[i + 1] = { method: "GET", relative_url: "me/" + edges[i] };
        }
        var self = this;
        FB.api('/', 'POST', {
            batch: batchArray
        }, function (responses) {
            if (responses.length && responses[0].body) {
                var response = JSON.parse(responses[0].body);
                self.pushRealtimeParameter("Facebook_ID", response.id);
                self.pushRealtimeParameter("Facebook_Name", response.name);
                self.pushRealtimeParameter("Facebook_FirstName", response.first_name);
                self.pushRealtimeParameter("Facebook_LastName", response.last_name);
                self.pushRealtimeParameter("Facebook_Is_Logged_In", "true");
                if (response.username) {
                    self.pushRealtimeParameter("Facebook_username", response.username);
                }
                if (response.gender) {
                    self.pushRealtimeParameter("Facebook_Gender", response.gender);
                }
                if (response.locale) {
                    self.pushRealtimeParameter("Facebook_Locale", response.locale);
                }
                if (response.email) {
                    self.pushRealtimeParameter("Facebook_Email", response.email);
                }
                if (response.birthday) {
                    self.pushRealtimeParameter("Facebook_Birthday", response.birthday);
                }
                if (response.location) {
                    self.pushRealtimeParameter("Facebook_Location", response.location.name);
                }
                if (response.hometown) {
                    self.pushRealtimeParameter("Facebook_Hometown", response.hometown.name);
                }
            }
            for (var i = 0; i < edges.length; i++) {
                if (responses.length >= i + 2 && responses[0].body && responses[i + 1].body) {
                    var response = JSON.parse(responses[i + 1].body);
                    if (response.data) {
                        var nameValues = '';
                        for (var j = 0; j < response.data.length; j++) {
                            if (response.data[j].name) {
                                if (nameValues != '') {
                                    nameValues = nameValues + '_';
                                }
                                nameValues = nameValues + response.data[j].name;
                            }
                        }
                        if (nameValues) {
                            if (edges[i] == 'likes') {
                                self.pushRealtimeParameter("Facebook_Likes", nameValues);
                            }
                            else if (edges[i] == 'events') {
                                self.pushRealtimeParameter("Facebook_Events", nameValues);
                            }
                            else {
                                self.pushRealtimeParameter("Facebook_" + edges[i], nameValues);
                            }
                        }
                    }
                }
            }
            if (!self.realtimeParameters.length) {
                self.pushRealtimeParameter("Facebook_Is_Logged_In", "false");
            }
            if (reload == true) {
                self.submitVisitorDetails(function () { location.reload(); });
            }
            else {
                self.submitVisitorDetails();
            }
        });
    };
    return RpiWebClient;
}());
var rpiWebClient = new RpiWebClient();
